#!/usr/bin/env python
# -*- coding: utf-8 -*-
# SPDX-License-Identifier: MIT
#
# MIT License
#
# Copyright (c) 2025 Bastiaan Williams
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#

"""
DeepL Automated Translation Script for Scribus
"""

import sys
import os
import json
from datetime import datetime
import urllib.request
import urllib.parse
import urllib.error

try:
    import scribus
except ImportError:
    print("This script must be run from inside Scribus.")
    sys.exit(1)

# Try Tkinter for the GUI
try:
    import tkinter as tk
    from tkinter import ttk
except ImportError:
    scribus.messageBox(
        "Error",
        "This script requires Tkinter (tkinter) for the GUI.\n"
        "Please install Tkinter and try again.",
        scribus.ICON_WARNING
    )
    sys.exit(1)

# -----------------------------------------------
# USER SETTINGS
#
# TIP:
# You can create a free account on DeepL.com.
# Their free API plan includes 500,000 characters per month,
# which is more than enough for most translation workflows.
#
# Add your DeepL API key below to enable automatic translation.
# -----------------------------------------------

DEEPL_API_KEY = "YOUR_DEEPL_API_KEY_HERE"
DEEPL_URL = "https://api-free.deepl.com/v2/translate"

LANGUAGE_CHOICES = [
    ("AR", "Arabic"),
    ("BG", "Bulgarian"),
    ("CS", "Czech"),
    ("DA", "Danish"),
    ("DE", "German"),
    ("EL", "Greek"),
    ("EN", "English"),
    ("EN-GB", "English (UK)"),
    ("EN-US", "English (US)"),
    ("ES", "Spanish"),
    ("ET", "Estonian"),
    ("FI", "Finnish"),
    ("FR", "French"),
    ("HE", "Hebrew"),
    ("HU", "Hungarian"),
    ("ID", "Indonesian"),
    ("IT", "Italian"),
    ("JA", "Japanese"),
    ("KO", "Korean"),
    ("LT", "Lithuanian"),
    ("LV", "Latvian"),
    ("NB", "Norwegian Bokmål"),
    ("NL", "Dutch"),
    ("PL", "Polish"),
    ("PT", "Portuguese"),
    ("PT-BR", "Portuguese (Brazil)"),
    ("RO", "Romanian"),
    ("RU", "Russian"),
    ("SK", "Slovak"),
    ("SL", "Slovenian"),
    ("SV", "Swedish"),
    ("TR", "Turkish"),
    ("UK", "Ukrainian"),
    ("ZH", "Chinese (Simplified)"),
]

LANG_LABEL_TO_CODE = {label: code for code, label in LANGUAGE_CHOICES}

translation_cache = {}


# -----------------------------
# IMPROVED SIMPLE GUI
# -----------------------------

def show_simple_gui():
    """Show a simple dialog using only tkinter widgets"""
    root = tk.Tk()
    root.title("Scribus DeepL Automated Translator")
    root.geometry("500x475")
    root.resizable(False, False)
    
    # Store results
    result = {
        "lang": None,
        "mode": None,
        "preview": True,
        "method": "create",
        "ok": False
    }
    
    # Variables
    lang_var = tk.StringVar()
    mode_var = tk.StringVar(value="selected")
    method_var = tk.StringVar(value="create")
    preview_var = tk.BooleanVar(value=True)
    
    # Set default language to English
    for code, label in LANGUAGE_CHOICES:
        if code == "EN":
            lang_var.set(label)
            break
    
    # Main container
    main_frame = tk.Frame(root)
    main_frame.pack(fill='both', expand=True, padx=20, pady=20)
    
    # Title
    tk.Label(main_frame, text="🌍 Scribus DeepL Automated Translator", 
             font=('Arial', 14, 'bold')).pack(pady=(0, 10))
    
    tk.Label(main_frame, text="translation the easy way....",
             font=('Arial', 10)).pack(pady=(0, 20))
    
    # Language selection - FIXED: Now uses Combobox which highlights selection
    tk.Label(main_frame, text="Target Language:", 
             font=('Arial', 10, 'bold')).pack(anchor='w', pady=(0, 5))
    
    # Use Combobox instead of Listbox for better selection highlighting
    lang_combo = ttk.Combobox(main_frame, textvariable=lang_var, state="readonly")
    lang_combo['values'] = [label for _, label in LANGUAGE_CHOICES]
    lang_combo.pack(fill='x', pady=(0, 15))
    
    # Translation Scope
    tk.Label(main_frame, text="Translation Scope:", 
             font=('Arial', 10, 'bold')).pack(anchor='w', pady=(0, 5))
    
    scope_frame = tk.Frame(main_frame)
    scope_frame.pack(fill='x', pady=(0, 15))
    
    tk.Radiobutton(scope_frame, text="📝 Selected frames", 
                   variable=mode_var, value="selected").pack(anchor='w')
    tk.Radiobutton(scope_frame, text="📄 Current page", 
                   variable=mode_var, value="page").pack(anchor='w')
    tk.Radiobutton(scope_frame, text="📖 Whole document", 
                   variable=mode_var, value="document").pack(anchor='w')
    
    # Translation Method
    tk.Label(main_frame, text="Translation Method:", 
             font=('Arial', 10, 'bold')).pack(anchor='w', pady=(0, 5))
    
    method_frame = tk.Frame(main_frame)
    method_frame.pack(fill='x', pady=(0, 15))
    
    tk.Radiobutton(method_frame, text="📑 CREATE NEW FRAMES (preserves formatting)", 
                   variable=method_var, value="create").pack(anchor='w')
    tk.Radiobutton(method_frame, text="📄 REPLACE TEXT (loses formatting)", 
                   variable=method_var, value="replace").pack(anchor='w')
    
    # Options
    options_frame = tk.Frame(main_frame)
    options_frame.pack(fill='x', pady=(0, 20))
    
    tk.Checkbutton(options_frame, text="Preview first", 
                   variable=preview_var).pack(anchor='w')
    
    # Buttons
    button_frame = tk.Frame(main_frame)
    button_frame.pack(fill='x', pady=(10, 0))
    
    def on_test():
        """Test DeepL connection"""
        test_text = "Hello, world! This is a test."
        params = {"text": test_text, "target_lang": "FR"}
        js = deepl_request(params)
        
        if js is None:
            scribus.messageBox(
                "DeepL Test",
                "❌ DeepL connection failed.\nPlease check your API key and internet connection.",
                scribus.ICON_WARNING
            )
            return
        
        try:
            translated = js["translations"][0]["text"]
            scribus.messageBox(
                "DeepL Test",
                f"✅ DeepL connection successful!\n\nOriginal: {test_text}\nTranslated: {translated}",
                scribus.ICON_INFORMATION
            )
        except Exception as e:
            scribus.messageBox(
                "DeepL Test Error",
                f"❌ Could not parse DeepL response:\n{str(e)}",
                scribus.ICON_WARNING
            )
    
    def on_about():
        """Show about dialog"""
        about_text = """Scribus DeepL Automated Translator

NOTE: NO guaranteed formatting preservation.
      (use the duplicate option to avoid format 
      of the original text frame(s))

HOW IT WORKS:
1. Creates NEW frames with translated text
2. Copies ALL formatting from original frames
3. Places new frames next to originals
4. Original frames remain unchanged

Benefits:
✅ ALL formatting preserved (fonts, styles, colors, spacing)
✅ Original text remains untouched
✅ Easy comparison side-by-side
✅ No risk of losing formatting

Version: 0.10 (10-12-2025)
Author: Bastiaan Williams
License: MIT"""
        scribus.messageBox(
            "About – Scribus DeepL Automated Translator",
            about_text,
            scribus.ICON_INFORMATION
        )
    
    def on_translate():
        """Handle translate button"""
        label = lang_var.get()
        if not label or label not in LANG_LABEL_TO_CODE:
            scribus.messageBox(
                "Error",
                "Please select a target language.",
                scribus.ICON_WARNING
            )
            return
        
        code = LANG_LABEL_TO_CODE[label]
        result["lang"] = code
        result["mode"] = mode_var.get()
        result["method"] = method_var.get()
        result["preview"] = preview_var.get()
        result["ok"] = True
        
        root.destroy()
    
    def on_cancel():
        """Handle cancel button"""
        root.destroy()
    
    # Button layout
    tk.Button(button_frame, text="About", command=on_about, 
              width=10).pack(side='left', padx=2)
    tk.Button(button_frame, text="Test DeepL", command=on_test, 
              width=10).pack(side='left', padx=2)
    tk.Button(button_frame, text="Cancel", command=on_cancel, 
              width=10).pack(side='right', padx=2)
    tk.Button(button_frame, text="Translate", command=on_translate, 
              width=12, bg="#4CAF50", fg="white").pack(side='right', padx=2)
    
    # Center window
    root.update_idletasks()
    width = root.winfo_width()
    height = root.winfo_height()
    x = (root.winfo_screenwidth() // 2) - (width // 2)
    y = (root.winfo_screenheight() // 2) - (height // 2)
    root.geometry(f'{width}x{height}+{x}+{y}')
    
    # Run mainloop
    root.mainloop()
    
    if not result["ok"]:
        return None, None, None, None
    
    return result["lang"], result["mode"], result["preview"], result["method"]


# -----------------------------
# UTILITY FUNCTIONS
# -----------------------------

def get_logfile_path():
    """Get a logfile path next to the current .sla document, or in CWD."""
    try:
        doc_path = scribus.getDocName()
    except Exception:
        doc_path = ""
    if not doc_path:
        return "scribus_translation_log.txt"
    base_dir = os.path.dirname(doc_path)
    if not base_dir:
        return "scribus_translation_log.txt"
    return os.path.join(base_dir, "scribus_translation_log.txt")


def write_log_header(logf, mode, tgt, preview, method):
    if logf is None:
        return
    now = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    logf.write("\n")
    logf.write("=== Scribus DeepL Automated Translation Run ===\n")
    logf.write("Time: {}\n".format(now))
    logf.write("Mode: {}\n".format(mode))
    logf.write("Target language: {}\n".format(tgt))
    logf.write("Method: {}\n".format(method))
    logf.write("Preview only: {}\n".format("YES" if preview else "NO"))
    logf.write("-------------------------------------\n")


def short_snippet(text, length=50):
    if not text:
        return ""
    t = text.strip().replace("\r", " ").replace("\n", " ")
    if len(t) <= length:
        return t
    return t[:length] + "..."


# -----------------------------
# HTTP / DEEPL HELPER
# -----------------------------

def deepl_request(params):
    """
    Low-level DeepL HTTP helper.
    Returns JSON dict on success,
    returns None on fatal error (401/403/429)
    and shows a message to the user.
    """
    if not DEEPL_API_KEY or DEEPL_API_KEY == "YOUR_DEEPL_API_KEY_HERE":
        scribus.messageBox(
            "DeepL Error",
            "Please set your DeepL API key in the script.",
            scribus.ICON_WARNING
        )
        return None

    params["auth_key"] = DEEPL_API_KEY
    data = urllib.parse.urlencode(params).encode("utf-8")

    try:
        req = urllib.request.Request(DEEPL_URL, data=data)
        f = urllib.request.urlopen(req)
        js = json.loads(f.read().decode("utf-8"))
        return js

    except urllib.error.HTTPError as e:
        code = e.code
        if code == 429:
            scribus.messageBox(
                "DeepL limit reached",
                "DeepL HTTP 429: Too many requests / quota used.\n"
                "Translation will stop.",
                scribus.ICON_WARNING
            )
            return None
        if code in (401, 403):
            scribus.messageBox(
                "DeepL authentication / permission error",
                (
                    "DeepL HTTP %d.\n"
                    "This usually means:\n"
                    "- Invalid API key\n"
                    "- Wrong endpoint (free vs pro)\n"
                    "- Or your account has no access.\n"
                    "Translation will stop."
                ) % code,
                scribus.ICON_WARNING
            )
            return None
        scribus.messageBox("DeepL HTTP Error", str(e), scribus.ICON_WARNING)
        return None

    except Exception as e:
        scribus.messageBox("DeepL Error", str(e), scribus.ICON_WARNING)
        return None


# -----------------------------
# TRANSLATION ENGINE 
# -----------------------------

def translate_text_auto_source(text, tgt):
    """
    Call DeepL for translation of one string.
    Let DeepL auto-detect the source language.
    Uses runtime cache to avoid duplicate calls.
    Returns None on fatal DeepL error.
    """
    if not text:
        return text

    tgt_code = tgt  # Already in correct format

    cache_key = (tgt_code, text)
    if cache_key in translation_cache:
        return translation_cache[cache_key]

    params = {
        "text": text,
        "target_lang": tgt_code
    }

    js = deepl_request(params)
    if js is None:
        return None

    try:
        translated = js["translations"][0]["text"]
        translation_cache[cache_key] = translated
        return translated
    except Exception as e:
        scribus.messageBox("DeepL Parse Error", str(e), scribus.ICON_WARNING)
        return text


# -----------------------------
# FORMATTING PRESERVATION AND FRAME PLACEMENT - NO ROTATION
# We try to keep the formatting a good as possible
# Be aware it is not perfect.
# -----------------------------

def get_frame_all_properties(frame_name):
    """Get ALL properties from a frame - BUT DON'T GET ROTATION"""
    properties = {}
    
    try:
        # Get basic frame properties
        x, y = scribus.getPosition(frame_name)
        width, height = scribus.getSize(frame_name)
        
        properties['x'] = x
        properties['y'] = y
        properties['width'] = width
        properties['height'] = height
        
        # DO NOT GET ROTATION - Keep new frames at 0 degrees
        
        # Get columns and gaps
        try:
            properties['columns'] = scribus.getColumns(frame_name)
        except:
            properties['columns'] = 1
            
        try:
            properties['column_gap'] = scribus.getColumnGap(frame_name)
        except:
            properties['column_gap'] = 0
            
        # Get alignment
        try:
            properties['alignment'] = scribus.getAlignment(frame_name)
        except:
            properties['alignment'] = 0  # Left
            
        # Get line spacing - THIS IS CRITICAL
        try:
            properties['line_spacing'] = scribus.getLineSpacing(frame_name)
        except:
            properties['line_spacing'] = 0
            
        # Get font properties
        try:
            properties['font'] = scribus.getFont(frame_name)
        except:
            properties['font'] = "Arial"
            
        try:
            properties['font_size'] = scribus.getFontSize(frame_name)
        except:
            properties['font_size'] = 12
            
        try:
            properties['text_color'] = scribus.getTextColor(frame_name)
        except:
            properties['text_color'] = "Black"
            
        # Get margins
        try:
            margins = scribus.getTextMargins(frame_name)
            properties['left_margin'] = margins[0]
            properties['right_margin'] = margins[1]
            properties['top_margin'] = margins[2]
            properties['bottom_margin'] = margins[3]
        except:
            properties['left_margin'] = properties['right_margin'] = 0
            properties['top_margin'] = properties['bottom_margin'] = 0
            
        # Get fill and line colors
        try:
            properties['fill_color'] = scribus.getFillColor(frame_name)
        except:
            properties['fill_color'] = "White"
            
        try:
            properties['line_color'] = scribus.getLineColor(frame_name)
        except:
            properties['line_color'] = "Black"
            
        try:
            properties['line_width'] = scribus.getLineWidth(frame_name)
        except:
            properties['line_width'] = 1
            
        # Get text flow mode
        try:
            properties['text_flow_mode'] = scribus.getTextFlowMode(frame_name)
        except:
            properties['text_flow_mode'] = 0
            
        # Get first line offset
        try:
            properties['first_line_offset'] = scribus.getFirstLineOffset(frame_name)
        except:
            properties['first_line_offset'] = 0
            
    except Exception as e:
        print(f"Error getting frame properties for {frame_name}: {e}")
    
    return properties


def apply_frame_all_properties(target_frame, properties):
    """Apply ALL properties to a frame - BUT DON'T SET ROTATION"""
    try:
        # Set size
        scribus.sizeObject(properties.get('width', 100), properties.get('height', 100), target_frame)
        
        # Position will be set separately in create_translated_frame
        
        # DO NOT SET ROTATION - Keep at 0 degrees
        # Instead, explicitly set rotation to 0
        try:
            scribus.setRotation(0, target_frame)
        except:
            pass
                
        # Set columns
        if 'columns' in properties:
            try:
                scribus.setColumns(properties['columns'], target_frame)
            except:
                pass
                
        if 'column_gap' in properties:
            try:
                scribus.setColumnGap(properties['column_gap'], target_frame)
            except:
                pass
                
        # Set alignment
        if 'alignment' in properties:
            try:
                scribus.setAlignment(properties['alignment'], target_frame)
            except:
                pass
                
        # Set line spacing - THIS IS CRITICAL FOR FIX
        if 'line_spacing' in properties:
            try:
                scribus.setLineSpacing(properties['line_spacing'], target_frame)
            except:
                # Try alternative method if available
                try:
                    scribus.setLineSpacing(properties['line_spacing'], target_frame)
                except:
                    print(f"Could not set line spacing for {target_frame}")
                    
        # Set font properties
        if 'font' in properties:
            try:
                scribus.setFont(properties['font'], target_frame)
            except:
                pass
                
        if 'font_size' in properties:
            try:
                scribus.setFontSize(properties['font_size'], target_frame)
            except:
                pass
                
        if 'text_color' in properties:
            try:
                scribus.setTextColor(properties['text_color'], target_frame)
            except:
                pass
                
        # Set margins
        if all(key in properties for key in ['left_margin', 'right_margin', 'top_margin', 'bottom_margin']):
            try:
                scribus.setTextMargins(
                    properties['left_margin'],
                    properties['right_margin'],
                    properties['top_margin'],
                    properties['bottom_margin'],
                    target_frame
                )
            except:
                pass
                
        # Set fill and line colors
        if 'fill_color' in properties:
            try:
                scribus.setFillColor(properties['fill_color'], target_frame)
            except:
                pass
                
        if 'line_color' in properties:
            try:
                scribus.setLineColor(properties['line_color'], target_frame)
            except:
                pass
                
        if 'line_width' in properties:
            try:
                scribus.setLineWidth(properties['line_width'], target_frame)
            except:
                pass
                
        # Set text flow mode
        if 'text_flow_mode' in properties:
            try:
                scribus.setTextFlowMode(properties['text_flow_mode'], target_frame)
            except:
                pass
                
        # Set first line offset
        if 'first_line_offset' in properties:
            try:
                scribus.setFirstLineOffset(properties['first_line_offset'], target_frame)
            except:
                pass
                
    except Exception as e:
        print(f"Error applying properties to {target_frame}: {e}")


def get_text_formatting(frame_name):
    """Get text formatting information"""
    try:
        text = scribus.getText(frame_name)
        if not text or len(text) == 0:
            return text, []
        
        text_length = len(text)
        
        # Sample at a few key points
        sample_points = [0]
        if text_length > 10:
            sample_points.append(min(10, text_length-1))
        if text_length > 50:
            sample_points.append(min(50, text_length-1))
        if text_length > 100:
            sample_points.append(min(100, text_length-1))
        
        # Always check last character
        if text_length > 1:
            sample_points.append(text_length - 1)
        
        formatting_info = []
        
        for pos in sample_points:
            try:
                scribus.selectText(pos, 1, frame_name)
                
                fmt = {
                    'position': pos,
                    'font': scribus.getFont(frame_name),
                    'font_size': scribus.getFontSize(frame_name),
                    'text_color': scribus.getTextColor(frame_name),
                    'bold': False,
                    'italic': False,
                    'underline': False
                }
                
                # Check bold
                try:
                    weight = scribus.getFontWeight(frame_name)
                    fmt['bold'] = (weight > 400)
                except:
                    pass
                    
                # Check italic
                try:
                    fmt['italic'] = scribus.getFontItalic(frame_name)
                except:
                    pass
                    
                # Check underline
                try:
                    fmt['underline'] = scribus.getFontUnderline(frame_name)
                except:
                    pass
                    
                formatting_info.append(fmt)
                
            except Exception as e:
                print(f"Error getting formatting at position {pos}: {e}")
                continue
        
        scribus.deselectAll()
        
        return text, formatting_info
        
    except Exception as e:
        print(f"Error getting text formatting: {e}")
        try:
            text = scribus.getText(frame_name)
            return text, []
        except:
            return "", []


def apply_text_formatting(target_frame, formatting_info):
    """Apply text formatting based on formatting info"""
    try:
        text_length = len(scribus.getText(target_frame))
        if text_length == 0 or not formatting_info:
            return
        
        # First, apply the first formatting to entire text
        first_fmt = formatting_info[0]
        
        # Select all text
        scribus.selectText(0, text_length, target_frame)
        
        # Apply font and size
        try:
            scribus.setFont(first_fmt['font'], target_frame)
        except:
            pass
            
        try:
            scribus.setFontSize(first_fmt['font_size'], target_frame)
        except:
            pass
            
        try:
            scribus.setTextColor(first_fmt['text_color'], target_frame)
        except:
            pass
        
        # Set to normal weight first (400)
        try:
            scribus.setFontWeight(400, target_frame)
        except:
            pass
            
        try:
            scribus.setFontItalic(False, target_frame)
        except:
            pass
            
        try:
            scribus.setFontUnderline(False, target_frame)
        except:
            pass
        
        scribus.deselectAll()
        
        # Now apply bold/italic/underline only where needed
        for fmt in formatting_info:
            # Apply from this position to end, or to next formatting change
            start_pos = fmt['position']
            if start_pos >= text_length:
                continue
                
            # Find end position
            end_pos = text_length
            next_fmt_positions = [f['position'] for f in formatting_info if f['position'] > start_pos]
            if next_fmt_positions:
                end_pos = min(next_fmt_positions[0], text_length)
            
            segment_length = end_pos - start_pos
            if segment_length > 0:
                scribus.selectText(start_pos, segment_length, target_frame)
                
                # Apply formatting if different from normal
                if fmt.get('bold', False):
                    try:
                        scribus.setFontWeight(700, target_frame)
                    except:
                        pass
                        
                if fmt.get('italic', False):
                    try:
                        scribus.setFontItalic(True, target_frame)
                    except:
                        pass
                        
                if fmt.get('underline', False):
                    try:
                        scribus.setFontUnderline(True, target_frame)
                    except:
                        pass
                        
                scribus.deselectAll()
                
    except Exception as e:
        print(f"Error applying text formatting: {e}")
        try:
            scribus.deselectAll()
        except:
            pass


def find_best_position(original_x, original_y, width, height, page):
    """Find the best position for a translated frame near the original"""
    try:
        # Get page size
        page_width, page_height = scribus.getPageSize()
        
        # Try positions in order of preference
        positions = [
            # Try to the right first (most common)
            (original_x + width + 20, original_y),
            # Try to the left
            (original_x - width - 20, original_y),
            # Try below
            (original_x, original_y + height + 20),
            # Try above
            (original_x, original_y - height - 20),
            # Try diagonally right-down
            (original_x + width + 20, original_y + height + 20),
            # Try diagonally left-up
            (original_x - width - 20, original_y - height - 20),
        ]
        
        # Check each position
        for new_x, new_y in positions:
            # Make sure it's on the page with some margin
            margin = 10
            if (margin <= new_x <= page_width - width - margin and 
                margin <= new_y <= page_height - height - margin):
                return new_x, new_y
        
        # If no ideal position found, try to fit it somewhere on the page
        # Center it on the page as last resort
        new_x = max(margin, min(page_width - width - margin, original_x))
        new_y = max(margin, min(page_height - height - margin, original_y))
        
        return new_x, new_y
        
    except Exception as e:
        print(f"Error finding best position: {e}")
        # Default to right of original
        return original_x + width + 20, original_y


def create_translated_frame(source_frame, translated_text, page):
    """Create a new frame with translated text and copied formatting - FIXED PLACEMENT"""
    try:
        print(f"DEBUG: Starting create_translated_frame for {source_frame}")
        
        # Get source frame properties
        properties = get_frame_all_properties(source_frame)
        original_x = properties.get('x', 0)
        original_y = properties.get('y', 0)
        width = properties.get('width', 100)
        height = properties.get('height', 100)
        
        print(f"DEBUG: Original frame at ({original_x}, {original_y}), size {width}x{height}")
        
        # Find the best position near the original frame
        new_x, new_y = find_best_position(original_x, original_y, width, height, page)
        print(f"DEBUG: New position at ({new_x}, {new_y})")
        
        # Create new frame at the calculated position
        new_frame = scribus.createText(new_x, new_y, width, height)
        print(f"DEBUG: Created new frame: {new_frame}")
        
        if not translated_text or not translated_text.strip():
            print(f"DEBUG: WARNING - Translated text is empty!")
        
        # Set the translated text
        scribus.setText(translated_text, new_frame)
        
        # Apply ALL properties from source
        apply_frame_all_properties(new_frame, properties)
        
        # Get and apply text formatting
        original_text, formatting_info = get_text_formatting(source_frame)
        if formatting_info:
            apply_text_formatting(new_frame, formatting_info)
        
        # Verify text was actually set
        check_text = scribus.getText(new_frame)
        if not check_text or not check_text.strip():
            print(f"DEBUG: ERROR - New frame has no text after creation!")
        
        # Set frame name
        try:
            scribus.setName(new_frame, f"{source_frame}_translated")
        except:
            pass
            
        print(f"DEBUG: Successfully created translated frame")
        return new_frame
        
    except Exception as e:
        print(f"Error creating translated frame: {e}")
        return None


# -----------------------------
# FRAME COLLECTION HELPERS 
# -----------------------------

def get_selected_text_frames_with_pages():
    """
    Return (frames, frame_pages) for selected text frames.
    FIXED: Properly gets ALL selected frames
    """
    frames = []
    frame_pages = {}
    
    # Get current page for reference
    current_page = scribus.currentPage()
    
    # Get all selected objects
    sel_count = scribus.selectionCount()
    if sel_count == 0:
        print("DEBUG: No frames selected")
        return frames, frame_pages

    print(f"DEBUG: Found {sel_count} selected objects")
    
    for i in range(sel_count):
        try:
            name = scribus.getSelectedObject(i)
            print(f"DEBUG: Processing selected object {i}: {name}")
            
            # Check if it's a text frame
            obj_type = scribus.getObjectType(name)
            print(f"DEBUG: Object type: {obj_type}")
            
            if obj_type == "TextFrame":
                frames.append(name)
                print(f"DEBUG: Added text frame: {name}")
                
                # Get page number for this frame
                page_num = find_frame_page(name, current_page)
                frame_pages[name] = page_num
                print(f"DEBUG: Frame {name} is on page {page_num}")
                
        except Exception as e:
            print(f"DEBUG: Error processing selected object {i}: {e}")
            continue

    print(f"DEBUG: Total text frames found: {len(frames)}")
    return frames, frame_pages


def find_frame_page(frame_name, start_page):
    """
    Find which page a frame is on by checking all pages
    """
    try:
        current_page = scribus.currentPage()
        
        # First check current page
        for item in scribus.getPageItems():
            if item[0] == frame_name:
                return current_page
        
        # If not found on current page, check all pages
        total_pages = scribus.pageCount()
        for page in range(1, total_pages + 1):
            if page == current_page:
                continue  # Already checked
            scribus.gotoPage(page)
            for item in scribus.getPageItems():
                if item[0] == frame_name:
                    # Return to original page
                    scribus.gotoPage(current_page)
                    return page
        
        # Return to original page
        scribus.gotoPage(current_page)
        return current_page  # Default to current page
        
    except Exception as e:
        print(f"Error finding frame page: {e}")
        return start_page


def get_all_text_frames_document():
    """Return (frames, frame_pages) for all pages in the document."""
    frames = []
    frame_pages = {}
    
    try:
        current_page = scribus.currentPage()
        total_pages = scribus.pageCount()
        
        for page in range(1, total_pages + 1):
            scribus.gotoPage(page)
            page_items = scribus.getPageItems()
            
            for item in page_items:
                if item[1] == 4:  # TextFrame type (integer 4 from getPageItems)
                    frames.append(item[0])
                    frame_pages[item[0]] = page
        
        # Return to original page
        scribus.gotoPage(current_page)
        
    except Exception as e:
        print(f"Error getting all frames: {e}")
    
    return frames, frame_pages


def get_frames_current_page():
    """Return (frames, frame_pages) for the current page."""
    frames = []
    frame_pages = {}
    
    try:
        page = scribus.currentPage()
        page_items = scribus.getPageItems()
        
        for item in page_items:
            if item[1] == 4:  # TextFrame type (integer 4 from getPageItems)
                frames.append(item[0])
                frame_pages[item[0]] = page
                
    except Exception as e:
        print(f"Error getting current page frames: {e}")
    
    return frames, frame_pages


# -----------------------------
# PROCESSING FUNCTIONS 
# -----------------------------


def process_frames_create_method(frames, frame_pages, tgt, preview, logf):
    """
    CREATE METHOD: Creates new frames with translated text
    """
    translated = 0
    skipped = 0
    fatal_error = False
    translated_words = 0
    preview_details = []
    
    if logf:
        logf.write("Using CREATE method (new frames with preserved formatting)\n")

    # Store original page
    original_page = scribus.currentPage()
    
    print(f"DEBUG: Starting to process {len(frames)} frames")
    
    for idx, frame_name in enumerate(frames):
        print(f"\nDEBUG: Processing frame {idx+1}/{len(frames)}: {frame_name}")
        
        page = frame_pages.get(frame_name, original_page)
        
        try:
            # Navigate to the correct page if needed
            current_page = scribus.currentPage()
            if page != current_page:
                try:
                    print(f"DEBUG: Navigating from page {current_page} to page {page}")
                    scribus.gotoPage(page)
                except Exception as nav_error:
                    print(f"DEBUG: Navigation error: {nav_error}")
                    # If can't navigate, use current page
                    page = current_page
            
            text = scribus.getText(frame_name)
            print(f"DEBUG: Retrieved text length: {len(text) if text else 0}")
            
        except Exception as e:
            print(f"Error accessing frame {frame_name}: {e}")
            skipped += 1
            continue

        if not text or not text.strip():
            print(f"DEBUG: Frame {frame_name} has no text, skipping")
            skipped += 1
            continue

        # Get word count for preview
        words = text.split()
        frame_word_count = len(words)
        snippet_orig = short_snippet(text)
        
        print(f"DEBUG: Frame {frame_name} has {frame_word_count} words: '{snippet_orig}'")

        if preview:
            # Preview: just show what would be translated
            new_text = translate_text_auto_source(text, tgt)
            if new_text is None:
                fatal_error = True
                print("DEBUG: Fatal DeepL error, stopping")
                break

            translated += 1
            translated_words += frame_word_count
            
            # Show preview
            orig_words = words[:3]
            new_words = new_text.split()[:3]
            mapping = " | ".join([f"{orig_words[i]} → {new_words[i]}" 
                                 for i in range(min(len(orig_words), len(new_words)))])
            
            preview_details.append((page, frame_name, frame_word_count, mapping))
            print(f"DEBUG: Preview added for frame {frame_name}")
            continue

        # REAL TRANSLATION - CREATE NEW FRAME
        try:
            print(f"DEBUG: Translating frame {frame_name}")
            # Translate the text
            new_text = translate_text_auto_source(text, tgt)
            if new_text is None:
                fatal_error = True
                print("DEBUG: Fatal DeepL error during translation, stopping")
                break
            
            print(f"DEBUG: Translation complete, creating translated frame for {frame_name}")
            print(f"DEBUG: Translated text length: {len(new_text) if new_text else 0}")
            
            # Create new frame with translated text
            new_frame = create_translated_frame(frame_name, new_text, page)
            
            if new_frame:
                translated += 1
                translated_words += frame_word_count
                
                # Log success
                if logf:
                    logf.write(f"[CREATED] page={page}, original='{frame_name}', new='{new_frame}'\n")
                    logf.write(f"   original: {snippet_orig}\n")
                    logf.write(f"   translated: {short_snippet(new_text)}\n")
                    
                print(f"DEBUG: Successfully created translated frame {new_frame}")
            else:
                print(f"DEBUG: Failed to create translated frame for {frame_name}")
                skipped += 1
                
        except Exception as e:
            print(f"Error processing {frame_name}: {e}")
            skipped += 1

    # Return to original page
    try:
        scribus.gotoPage(original_page)
    except:
        pass

    print(f"DEBUG: Finished processing. Translated: {translated}, Skipped: {skipped}")
    return translated, skipped, fatal_error, translated_words, preview_details


def process_frames_replace_method(frames, frame_pages, tgt, preview, logf):
    """
    REPLACE METHOD: Replaces text in place (loses formatting)
    Fast but loses all formatting
    """
    translated = 0
    skipped = 0
    fatal_error = False
    translated_words = 0
    preview_details = []
    
    if logf:
        logf.write("Using REPLACE method (fast but loses formatting)\n")

    for frame_name in frames:
        page = frame_pages.get(frame_name, 0)
        
        try:
            text = scribus.getText(frame_name)
        except Exception:
            text = ""

        if not text or not text.strip():
            skipped += 1
            continue

        words = text.split()
        frame_word_count = len(words)
        snippet_orig = short_snippet(text)

        if preview:
            # Preview only
            new_text = translate_text_auto_source(text, tgt)
            if new_text is None:
                fatal_error = True
                break

            translated += 1
            translated_words += frame_word_count
            
            preview_details.append((page, frame_name, frame_word_count, "Replace method"))
            continue

        # REAL TRANSLATION - REPLACE IN PLACE
        try:
            new_text = translate_text_auto_source(text, tgt)
            if new_text is None:
                fatal_error = True
                break
            
            # Replace text (this WILL lose formatting)
            scribus.setText(new_text, frame_name)
            
            translated += 1
            translated_words += frame_word_count
            
            if logf:
                logf.write(f"[REPLACED] page={page}, frame='{frame_name}'\n")
                logf.write(f"   original: {snippet_orig}\n")
                logf.write(f"   translated: {short_snippet(new_text)}\n")
                
        except Exception as e:
            print(f"Error processing {frame_name}: {e}")
            skipped += 1

    return translated, skipped, fatal_error, translated_words, preview_details


# -----------------------------
# MAIN - SIMPLIFIED WITHOUT MULTIPLE TRANSLATIONS
# -----------------------------

def main():
    if not scribus.haveDoc():
        scribus.messageBox("Error", "Open a document first.", scribus.ICON_WARNING)
        return

    # Get user options using simple GUI
    try:
        tgt, mode, preview, method = show_simple_gui()
    except Exception as e:
        scribus.messageBox("GUI Error", f"Could not show GUI: {e}", scribus.ICON_WARNING)
        return
    
    if not tgt or not mode:
        return

    # Collect frames
    if mode == "page":
        frames, frame_pages = get_frames_current_page()
        mode_label = "Current page"
    elif mode == "selected":
        frames, frame_pages = get_selected_text_frames_with_pages()
        mode_label = "Selected frames"
    elif mode == "document":
        frames, frame_pages = get_all_text_frames_document()
        mode_label = "Whole document"
    else:
        return

    if not frames:
        scribus.messageBox("No Frames Found", 
                          "No text frames were found.\n\n"
                          "Please select some text frames and try again.",
                          scribus.ICON_WARNING)
        return

    # Open logfile
    log_path = get_logfile_path()
    try:
        logf = open(log_path, "a", encoding="utf-8")
    except Exception:
        logf = None

    write_log_header(logf, mode_label, tgt, preview, method)

    # Process frames based on selected method
    if method == "create":
        translated, skipped, fatal, translated_words, preview_details = process_frames_create_method(
            frames, frame_pages, tgt, preview, logf
        )
    else:  # replace method
        translated, skipped, fatal, translated_words, preview_details = process_frames_replace_method(
            frames, frame_pages, tgt, preview, logf
        )

    if logf:
        try:
            logf.write(f"Summary: translated_frames={translated}, skipped_frames={skipped}, "
                      f"translated_words={translated_words}, fatal_error={fatal}\n")
            logf.write("-------------------------------------\n")
            logf.close()
        except Exception:
            pass

    # Preview flow
    if preview:
        if fatal:
            msg = (f"Preview stopped due to DeepL error.\n\n"
                  f"Frames to process: {translated}\n"
                  f"Words to translate: {translated_words}\n"
                  f"Frames skipped: {skipped}\n\n"
                  f"Check logfile: {log_path}")
            scribus.messageBox("Preview (error)", msg, scribus.ICON_WARNING)
            return

        if translated == 0:
            scribus.messageBox("Preview", "No frames to translate.", scribus.ICON_INFORMATION)
            return

        # Ask to apply
        method_note = "CREATE NEW FRAMES" if method == "create" else "REPLACE TEXT (loses formatting)"
        
        res = scribus.messageBox(
            f"Apply translations? ({method_note})",
            f"Do you want to apply these translations?\n\n"
            f"Method: {method_note}\n"
            f"Frames to translate: {translated}\n"
            f"Words to translate: ~{translated_words}\n"
            f"Empty frames skipped: {skipped}\n\n"
            f"{'✅ Formatting will be PRESERVED with new frames' if method == 'create' else '⚠️  WARNING: All formatting will be LOST!'}",
            scribus.ICON_INFORMATION,
            scribus.BUTTON_YES,
            scribus.BUTTON_NO
        )

        if res != scribus.BUTTON_YES:
            scribus.messageBox("Translation Cancelled", 
                              "Translation was cancelled.\n\nScript will now exit.",
                              scribus.ICON_INFORMATION)
            return

        # Apply translations
        try:
            logf2 = open(log_path, "a", encoding="utf-8")
        except Exception:
            logf2 = None

        write_log_header(logf2, f"{mode_label} (APPLY)", tgt, False, method)

        if method == "create":
            translated2, skipped2, fatal2, translated_words2, _ = process_frames_create_method(
                frames, frame_pages, tgt, False, logf2
            )
        else:
            translated2, skipped2, fatal2, translated_words2, _ = process_frames_replace_method(
                frames, frame_pages, tgt, False, logf2
            )

        if logf2:
            try:
                logf2.write(f"Summary (apply): translated_frames={translated2}, "
                           f"skipped_frames={skipped2}, translated_words={translated_words2}\n")
                logf2.close()
            except Exception:
                pass

        msg = (f"Translation completed!\n\n"
              f"Method: {method_note}\n"
              f"Frames translated: {translated2}\n"
              f"Words translated: {translated_words2}\n"
              f"Frames skipped: {skipped2}\n\n")
        
        if method == "create":
            msg += "✅ New frames created with preserved formatting\n"
            msg += "✅ Original frames remain unchanged\n"
            msg += "✅ Translated frames placed to the right of originals\n"
        else:
            msg += "⚠️  Original text replaced (formatting lost)\n"
            
        msg += f"\nDetails: {log_path}"
        
        scribus.messageBox("Translation Complete", msg, scribus.ICON_INFORMATION)
        return

    # Direct translate flow (no preview)
    if fatal:
        msg = (f"Translation finished with error.\n\n"
              f"Frames translated: {translated}\n"
              f"Words translated: {translated_words}\n"
              f"Frames skipped: {skipped}\n\n"
              f"Details: {log_path}")
        
        scribus.messageBox("Translation Error", msg, scribus.ICON_WARNING)
        return
    else:
        method_note = "CREATE NEW FRAMES" if method == "create" else "REPLACE TEXT"
        msg = (f"Translation completed!\n\n"
              f"Method: {method_note}\n"
              f"Frames translated: {translated}\n"
              f"Words translated: {translated_words}\n"
              f"Frames skipped: {skipped}\n")
        
        if method == "create":
            msg += "\n✅ New frames created with preserved formatting\n"
            msg += "✅ Original frames remain unchanged\n"
            msg += "✅ Translated frames placed to the right of originals\n"
        else:
            msg += "\n⚠️  Original text replaced (formatting lost)\n"
            
        msg += f"\nDetails: {log_path}"
        
        scribus.messageBox("Translation Complete", msg, scribus.ICON_INFORMATION)


if __name__ == "__main__":
    main()
