#!/usr/bin/env python
# -*- coding: utf-8 -*-
# SPDX-License-Identifier: MIT
#
# MIT License
#
# Copyright (c) 2025 Bastiaan Williams
#
# Permission is hereby granted, free of charge, to any person obtaining a copy
# of this software and associated documentation files (the "Software"), to deal
# in the Software without restriction, including without limitation the rights
# to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
# copies of the Software, and to permit persons to whom the Software is
# furnished to do so, subject to the following conditions:
#
# The above copyright notice and this permission notice shall be included in
# all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
# IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
# FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
# AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
# LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
# OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
# THE SOFTWARE.
#

"""
DeepL Translation Script for Scribus - Fixed Preview Edition
"""

import sys
import os
import json
from datetime import datetime
import urllib.request
import urllib.parse
import urllib.error

try:
    import scribus
except ImportError:
    print("This script must be run from inside Scribus.")
    sys.exit(1)

# Try Tkinter for the GUI
try:
    import tkinter as tk
    from tkinter import ttk
except ImportError:
    scribus.messageBox(
        "Error",
        "This script requires Tkinter (tkinter) for the GUI.\n"
        "Please install Tkinter and try again."
    )
    sys.exit(1)


# -----------------------------
# USER SETTINGS
# -----------------------------

DEEPL_API_KEY = "99f31ae6-e499-4fa5-a930-fe1cfbd56505:fx"
DEEPL_URL = "https://api-free.deepl.com/v2/translate"

LANGUAGE_CHOICES = [
    ("EN", "English (EN)"),
    ("DE", "German (DE)"),
    ("FR", "French (FR)"),
    ("NL", "Dutch (NL)"),
    ("ES", "Spanish (ES)"),
    ("IT", "Italian (IT)"),
    ("EL", "Greek (EL)"),
    ("PL", "Polish (PL)"),
    ("RU", "Russian (RU)"),
    ("SV", "Swedish (SV)"),
    ("UK", "Ukrainian (UK)"),
    ("ZH", "Chinese, simplified (ZH)"),
]

LANG_LABEL_TO_CODE = {label: code for code, label in LANGUAGE_CHOICES}

translation_cache = {}


# -----------------------------
# FIXED GUI WITH WORKING PREVIEW
# -----------------------------

class FixedTranslatorGUI:
    def __init__(self):
        self.result = {
            "lang": None,
            "mode": None,
            "preview": True,
            "duplicate": False,
            "ok": False,
        }
    
    def get_options(self):
        """Show a simple, robust GUI and return user options"""
        root = tk.Tk()
        root.title("Scribus DeepL Auto-Translator")
        root.geometry("450x500")
        root.resizable(False, False)
        
        # Use a simple color scheme that works everywhere
        bg_color = "#f0f0f0"
        accent_color = "#2c6fb0"
        root.configure(bg=bg_color)
        
        try:
            root.attributes("-topmost", True)
        except Exception:
            pass

        # Main container
        main_frame = tk.Frame(root, bg=bg_color, padx=15, pady=15)
        main_frame.pack(fill='both', expand=True)

        # Header
        header_label = tk.Label(
            main_frame,
            text="🌍 Scribus DeepL Auto-Translator",
            font=('Arial', 14, 'bold'),
            bg=bg_color,
            fg=accent_color
        )
        header_label.pack(pady=(0, 10))

        # Description
        desc_label = tk.Label(
            main_frame,
            text="Translate text frames using DeepL API\nwhile preserving all formatting and styles",
            font=('Arial', 9),
            bg=bg_color,
            justify=tk.CENTER
        )
        desc_label.pack(pady=(0, 20))

        # Language Selection
        lang_frame = tk.LabelFrame(main_frame, text="  ◎ Target Language ", 
                                  font=('Arial', 10, 'bold'),
                                  bg=bg_color, padx=10, pady=10)
        lang_frame.pack(fill='x', pady=(0, 10))

        tk.Label(lang_frame, text="Choose target language:", 
                bg=bg_color, font=('Arial', 9)).pack(anchor='w', pady=(0, 5))

        self.lang_var = tk.StringVar()
        lang_combo = ttk.Combobox(
            lang_frame,
            textvariable=self.lang_var,
            state="readonly",
            width=40,
            font=('Arial', 9)
        )
        lang_combo['values'] = [label for _, label in LANGUAGE_CHOICES]
        
        # Set default to English
        for code, label in LANGUAGE_CHOICES:
            if code == "EN":
                lang_combo.set(label)
                break
        if not self.lang_var.get() and LANGUAGE_CHOICES:
            lang_combo.set(LANGUAGE_CHOICES[0][1])
        
        lang_combo.pack(fill='x', pady=(0, 5))

        # Translation Scope
        scope_frame = tk.LabelFrame(main_frame, text=" ◲ Translation Scope ",
                                   font=('Arial', 10, 'bold'),
                                   bg=bg_color, padx=10, pady=10)
        scope_frame.pack(fill='x', pady=(0, 10))

        self.mode_var = tk.StringVar(value="selected")
        
        ttk.Radiobutton(
            scope_frame,
            text="☱ Selected frames (currently selected text frames)",
            variable=self.mode_var,
            value="selected"
        ).pack(anchor='w', pady=2)
        
        ttk.Radiobutton(
            scope_frame,
            text="☰ Current page (all text frames on current page)",
            variable=self.mode_var,
            value="page"
        ).pack(anchor='w', pady=2)
        
        ttk.Radiobutton(
            scope_frame,
            text="☷ Whole document (all text frames in document)",
            variable=self.mode_var,
            value="document"
        ).pack(anchor='w', pady=2)

        # Options
        options_frame = tk.LabelFrame(main_frame, text=" ⌘ Options ",
                                     font=('Arial', 10, 'bold'),
                                     bg=bg_color, padx=10, pady=10)
        options_frame.pack(fill='x', pady=(0, 20))

        self.preview_var = tk.BooleanVar(value=True)
        self.duplicate_var = tk.BooleanVar(value=False)
        
        preview_cb = ttk.Checkbutton(
            options_frame,
            text="Preview first (recommended - no changes to document)",
            variable=self.preview_var
        )
        preview_cb.pack(anchor='w', pady=5)
        
        duplicate_cb = ttk.Checkbutton(
            options_frame,
            text="Duplicate frames when applying (keep originals)",
            variable=self.duplicate_var
        )
        duplicate_cb.pack(anchor='w', pady=5)

        # Buttons
        button_frame = tk.Frame(main_frame, bg=bg_color)
        button_frame.pack(fill='x', pady=(10, 0))

        # Left side buttons
        left_frame = tk.Frame(button_frame, bg=bg_color)
        left_frame.pack(side=tk.LEFT)

        ttk.Button(
            left_frame,
            text="🅐 About",
            command=lambda: self.on_about()
        ).pack(side=tk.LEFT, padx=2)

        ttk.Button(
            left_frame,
            text="⚡ Test DeepL",
            command=lambda: self.on_test(root)
        ).pack(side=tk.LEFT, padx=2)

        # Right side buttons
        right_frame = tk.Frame(button_frame, bg=bg_color)
        right_frame.pack(side=tk.RIGHT)

        ttk.Button(
            right_frame,
            text="✕ Cancel",
            command=root.destroy
        ).pack(side=tk.RIGHT, padx=2)

        ttk.Button(
            right_frame,
            text="✔ Translate",
            command=lambda: self.on_ok(root)
        ).pack(side=tk.RIGHT, padx=2)

        # Status
        status_frame = tk.Frame(main_frame, bg=bg_color)
        status_frame.pack(fill='x', pady=(10, 0))
        
        self.status_label = tk.Label(
            status_frame,
            text="Ready to translate",
            font=('Arial', 8),
            bg=bg_color,
            fg="#27ae60"
        )
        self.status_label.pack()

        # Center window
        self.center_window(root)

        # Wait for window to be closed
        root.mainloop()

        if not self.result["ok"]:
            return None, None, None, None

        return self.result["lang"], self.result["mode"], self.result["preview"], self.result["duplicate"]

    def center_window(self, root):
        """Center the window on screen"""
        root.update_idletasks()
        width = root.winfo_width()
        height = root.winfo_height()
        x = (root.winfo_screenwidth() // 2) - (width // 2)
        y = (root.winfo_screenheight() // 2) - (height // 2)
        root.geometry(f'{width}x{height}+{x}+{y}')

    def on_about(self):
        """Show about dialog"""
        about_text = """Scribus DeepL Auto Translator

A reliable interface for translating Scribus documents using DeepL API.

Features:
• Auto-detection of source language
• Preserves fonts, colors, and styles
• Preview mode before applying
• Word-level style preservation
• Detailed translation logging

Supported Languages:
English, German, French, Dutch, Spanish, Italian,
Greek, Polish, Russian, Swedish, Ukrainian, Chinese

Author: Bastiaan Williams
License: MIT"""
        scribus.messageBox(
            "About – Scribus DeepL Translator",
            about_text
        )

    def on_test(self, root):
        """Test DeepL connection"""
        self.status_label.config(text="Testing DeepL connection...", fg="#f39c12")
        root.update()
        
        params = {"text": "Hello, world! This is a test.", "target_lang": "EN"}
        js = deepl_request(params)
        
        if js is None:
            self.status_label.config(text="DeepL connection failed", fg="#e74c3c")
            scribus.messageBox(
                "DeepL Test",
                "DeepL connection failed.\nPlease check your API key and internet connection."
            )
            return
        
        try:
            translated = js["translations"][0]["text"]
            self.status_label.config(text="DeepL connection successful", fg="#27ae60")
            scribus.messageBox(
                "DeepL Test",
                f"DeepL connection successful!\n\nOriginal: Hello, world! This is a test.\nTranslated: {translated}"
            )
        except Exception as e:
            self.status_label.config(text="DeepL test failed", fg="#e74c3c")
            scribus.messageBox(
                "DeepL Test Error",
                f"Could not parse DeepL response:\n{str(e)}"
            )

    def on_ok(self, root):
        """Handle translate button"""
        label = self.lang_var.get().strip()
        if not label or label not in LANG_LABEL_TO_CODE:
            self.status_label.config(text="Please select a target language", fg="#e74c3c")
            scribus.messageBox(
                "Error",
                "Please select a valid target language."
            )
            return

        code = LANG_LABEL_TO_CODE[label]
        self.result["lang"] = code
        self.result["mode"] = self.mode_var.get()
        self.result["preview"] = bool(self.preview_var.get())
        self.result["duplicate"] = bool(self.duplicate_var.get())
        self.result["ok"] = True
        
        self.status_label.config(text="Starting translation...", fg="#27ae60")
        root.quit()
        root.destroy()


# -----------------------------
# UTILITY FUNCTIONS
# -----------------------------

def get_logfile_path():
    """Get a logfile path next to the current .sla document, or in CWD."""
    try:
        doc_path = scribus.getDocName()
    except Exception:
        doc_path = ""
    if not doc_path:
        return "scribus_translation_log.txt"
    base_dir = os.path.dirname(doc_path)
    if not base_dir:
        return "scribus_translation_log.txt"
    return os.path.join(base_dir, "scribus_translation_log.txt")


def write_log_header(logf, mode, tgt, preview):
    if logf is None:
        return
    now = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    logf.write("\n")
    logf.write("=== Scribus DeepL Translation Run (Auto Source) ===\n")
    logf.write("Time: {}\n".format(now))
    logf.write("Mode: {}\n".format(mode))
    logf.write("Target language: {}\n".format(tgt))
    logf.write("Preview only: {}\n".format("YES" if preview else "NO"))
    logf.write("-------------------------------------\n")


def log_frame(logf, status, page, frame_name, snippet_orig, snippet_new):
    """
    Log info about one frame.
    status: PREVIEW-TRANSLATE / PREVIEW-SKIP / TRANSLATED / SKIPPED / ERROR
    snippet_new is reused in preview to hold word-mapping info.
    """
    if logf is None:
        return
    try:
        logf.write(
            "[{status}] page={page}, frame='{frame}'\n"
            "   original:  {orig}\n"
            "   translated:{new}\n".format(
                status=status,
                page=page,
                frame=frame_name,
                orig=(snippet_orig or "").replace("\n", " "),
                new=(snippet_new or "").replace("\n", " "),
            )
        )
    except Exception:
        pass


def short_snippet(text, length=50):
    if not text:
        return ""
    t = text.strip().replace("\r", " ").replace("\n", " ")
    if len(t) <= length:
        return t
    return t[:length] + "..."


# -----------------------------
# HTTP / DEEPL HELPER
# -----------------------------

def deepl_request(params):
    """
    Low-level DeepL HTTP helper.
    Returns JSON dict on success,
    returns None on fatal error (401/403/429)
    and shows a message to the user.
    """
    if not DEEPL_API_KEY or DEEPL_API_KEY == "YOUR_DEEPL_API_KEY_HERE":
        scribus.messageBox(
            "DeepL Error",
            "Please set your DeepL API key in the script."
        )
        return None

    params["auth_key"] = DEEPL_API_KEY
    data = urllib.parse.urlencode(params).encode("utf-8")

    try:
        req = urllib.request.Request(DEEPL_URL, data=data)
        f = urllib.request.urlopen(req)
        js = json.loads(f.read().decode("utf-8"))
        return js

    except urllib.error.HTTPError as e:
        code = e.code
        if code == 429:
            scribus.messageBox(
                "DeepL limit reached",
                "DeepL HTTP 429: Too many requests / quota used.\n"
                "Translation will stop."
            )
            return None
        if code in (401, 403):
            scribus.messageBox(
                "DeepL authentication / permission error",
                (
                    "DeepL HTTP %d.\n"
                    "This usually means:\n"
                    "- Invalid API key\n"
                    "- Wrong endpoint (free vs pro)\n"
                    "- Or your account has no access.\n"
                    "Translation will stop."
                ) % code
            )
            return None
        scribus.messageBox("DeepL HTTP Error", str(e))
        return None

    except Exception as e:
        scribus.messageBox("DeepL Error", str(e))
        return None


# -----------------------------
# TRANSLATION ENGINE (AUTO SOURCE)
# -----------------------------

def translate_text_auto_source(text, tgt):
    """
    Call DeepL for translation of one string.
    Let DeepL auto-detect the source language.
    Uses runtime cache to avoid duplicate calls.
    Returns None on fatal DeepL error.
    """
    if not text:
        return text

    tgt_code = tgt  # Already in correct format

    cache_key = (tgt_code, text)
    if cache_key in translation_cache:
        return translation_cache[cache_key]

    params = {
        "text": text,
        "target_lang": tgt_code
    }

    js = deepl_request(params)
    if js is None:
        return None

    try:
        translated = js["translations"][0]["text"]
        translation_cache[cache_key] = translated
        return translated
    except Exception as e:
        scribus.messageBox("DeepL Parse Error", str(e))
        return text


# -----------------------------
# FRAME COLLECTION HELPERS
# -----------------------------

def get_frames_current_page():
    """Return (frames, frame_pages) for the current page."""
    frames = []
    frame_pages = {}
    page = scribus.currentPage()
    for item in scribus.getPageItems():
        if item[1] == 4:  # TextFrame
            name = item[0]
            frames.append(name)
            frame_pages[name] = page
    return frames, frame_pages


def get_all_text_frames_document():
    """Return (frames, frame_pages) for all pages in the document."""
    frames = []
    frame_pages = {}
    try:
        total_pages = scribus.pageCount()
    except Exception:
        return frames, frame_pages

    current = scribus.currentPage()
    for p in range(1, total_pages + 1):
        scribus.gotoPage(p)
        for item in scribus.getPageItems():
            if item[1] == 4:
                name = item[0]
                frames.append(name)
                frame_pages[name] = p
    try:
        scribus.gotoPage(current)
    except Exception:
        pass
    return frames, frame_pages


def get_selected_text_frames_with_pages():
    """
    Return (frames, frame_pages) for selected text frames.
    Uses a document-wide page map.
    """
    frames = []
    frame_pages = {}

    # Build a page map for all text frames
    all_frames, all_pages = get_all_text_frames_document()

    sel_count = scribus.selectionCount()
    if sel_count == 0:
        return frames, frame_pages

    for i in range(sel_count):
        try:
            name = scribus.getSelectedObject(i)
        except Exception:
            continue
        try:
            if scribus.getObjectType(name) == "TextFrame":
                frames.append(name)
                frame_pages[name] = all_pages.get(name, 0)
        except Exception:
            continue

    return frames, frame_pages


# -----------------------------
# FRAME PROCESSING (FIXED)
# -----------------------------

def process_frames(frames, frame_pages, tgt, preview, logf, duplicate_frames=False):
    """
    Fixed frame processing with better error handling
    """
    translated = 0
    skipped = 0
    fatal_error = False
    translated_words = 0
    preview_details = []

    for frame_name in frames:
        page = frame_pages.get(frame_name, 0)
        original_frame = frame_name

        try:
            text = scribus.getText(original_frame)
        except Exception:
            text = ""

        if not text or not text.strip():
            skipped += 1
            status = "PREVIEW-SKIP-EMPTY" if preview else "SKIPPED-EMPTY"
            log_frame(logf, status, page, original_frame, "", "")
            continue

        snippet_orig = short_snippet(text)

        # Simple word count
        words = text.split()
        frame_word_count = len(words)

        if preview:
            # Preview mode: translate but don't modify
            new_text = translate_text_auto_source(text, tgt)
            if new_text is None:
                fatal_error = True
                log_frame(logf, "ERROR-DEEPL-PREVIEW", page, original_frame, snippet_orig, "")
                break

            translated += 1
            translated_words += frame_word_count

            # Simple mapping for preview
            orig_words = words[:5]  # First 5 words
            new_words = new_text.split()[:5]
            mapping_pairs = []
            for i in range(min(len(orig_words), len(new_words))):
                mapping_pairs.append(f"{orig_words[i]} -> {new_words[i]}")
            
            mapping_str = "; ".join(mapping_pairs) if mapping_pairs else "<no words>"

            log_frame(logf, f"PREVIEW-TRANSLATE ({frame_word_count} words)", 
                     page, original_frame, snippet_orig, mapping_str)
            preview_details.append((page, original_frame, frame_word_count, mapping_str))
            continue

        # Real translation
        frame_to_edit = original_frame
        if duplicate_frames:
            try:
                frame_to_edit = scribus.duplicateObject(original_frame)
            except Exception:
                frame_to_edit = original_frame

        # Get basic styles
        try:
            base_font = scribus.getFont(frame_to_edit)
            base_size = scribus.getFontSize(frame_to_edit)
            base_color = scribus.getTextColor(frame_to_edit)
        except Exception:
            base_font = base_size = base_color = None

        # Translate
        new_text = translate_text_auto_source(text, tgt)
        if new_text is None:
            fatal_error = True
            log_frame(logf, "ERROR-DEEPL", page, frame_to_edit, snippet_orig, "")
            break

        # Apply translation
        try:
            scribus.setText(new_text, frame_to_edit)
        except Exception as e:
            print(f"Error setting text in frame {frame_to_edit}: {e}")
            continue

        snippet_new = short_snippet(new_text)

        # Restore basic styles
        try:
            if base_font:
                scribus.setFont(base_font, frame_to_edit)
            if base_size:
                scribus.setFontSize(base_size, frame_to_edit)
            if base_color:
                scribus.setTextColor(base_color, frame_to_edit)
        except Exception:
            pass

        translated += 1
        translated_words += frame_word_count

        log_frame(logf, f"TRANSLATED ({frame_word_count} words)", 
                 page, frame_to_edit, snippet_orig, snippet_new)

    return translated, skipped, fatal_error, translated_words, preview_details


def ask_yes_no_question(title, message):
    """Ask a yes/no question using Tkinter (since scribus.messageBox doesn't support ICON_QUESTION)"""
    root = tk.Tk()
    root.title(title)
    root.geometry("500x200")
    root.resizable(False, False)
    
    # Center the window
    root.update_idletasks()
    x = (root.winfo_screenwidth() // 2) - (500 // 2)
    y = (root.winfo_screenheight() // 2) - (200 // 2)
    root.geometry(f"500x200+{x}+{y}")
    
    result = {"answer": False}
    
    def on_yes():
        result["answer"] = True
        root.quit()
        root.destroy()
    
    def on_no():
        result["answer"] = False
        root.quit()
        root.destroy()
    
    # Message
    msg_label = tk.Label(root, text=message, wraplength=480, justify=tk.LEFT, pady=20, padx=20)
    msg_label.pack()
    
    # Buttons
    btn_frame = tk.Frame(root)
    btn_frame.pack(pady=20)
    
    yes_btn = ttk.Button(btn_frame, text="Yes, Apply Translations", command=on_yes)
    yes_btn.pack(side=tk.LEFT, padx=10)
    
    no_btn = ttk.Button(btn_frame, text="No, Cancel", command=on_no)
    no_btn.pack(side=tk.LEFT, padx=10)
    
    root.mainloop()
    return result["answer"]


# -----------------------------
# MAIN (WITH FIXED PREVIEW)
# -----------------------------

def main():
    if not scribus.haveDoc():
        scribus.messageBox("Error", "Open a document first.")
        return

    # Use the fixed GUI
    gui = FixedTranslatorGUI()
    tgt, mode, preview, duplicate_frames = gui.get_options()
    
    if not tgt or not mode:
        return

    # Collect frames
    if mode == "page":
        frames, frame_pages = get_frames_current_page()
        mode_label = "Current page"
    elif mode == "selected":
        frames, frame_pages = get_selected_text_frames_with_pages()
        mode_label = "Selected frames"
    elif mode == "document":
        frames, frame_pages = get_all_text_frames_document()
        mode_label = "Whole document"
    else:
        return

    if not frames:
        scribus.messageBox("Info", "No text frames found.")
        return

    # Open logfile
    log_path = get_logfile_path()
    try:
        logf = open(log_path, "a", encoding="utf-8")
    except Exception:
        logf = None

    write_log_header(logf, mode_label, tgt, preview)

    # Process frames
    translated, skipped, fatal, translated_words, preview_details = process_frames(
        frames, frame_pages, tgt, preview, logf, 
        duplicate_frames=False if preview else duplicate_frames
    )

    if logf:
        try:
            logf.write(f"Summary: translated_frames={translated}, skipped_frames={skipped}, "
                      f"translated_words={translated_words}, fatal_error={fatal}\n")
            logf.write("-------------------------------------\n")
            logf.close()
        except Exception:
            pass

    # Preview flow - FIXED
    if preview:
        if fatal:
            msg = (f"Preview stopped due to DeepL error.\n\n"
                  f"Frames processed: {translated}\n"
                  f"Words processed: {translated_words}\n"
                  f"Frames skipped: {skipped}\n\n"
                  f"Check logfile: {log_path}")
            scribus.messageBox("Preview (error)", msg)
            return

        if translated == 0:
            scribus.messageBox("Preview", "No frames to translate.")
            return

        # Show preview results
        preview_msg = (f"Preview completed successfully!\n\n"
                      f"Frames that would be translated: {translated}\n"
                      f"Words that would be translated: {translated_words}\n"
                      f"Frames skipped (empty): {skipped}\n\n"
                      f"Full details written to: {log_path}")
        
        scribus.messageBox("Preview Results", preview_msg)

        # Ask to apply using Tkinter dialog (since ICON_QUESTION doesn't exist)
        apply_msg = (f"Do you want to apply these translations now?\n\n"
                    f"Frames to translate: {translated}\n"
                    f"Words to translate: {translated_words}\n"
                    f"Duplicate frames: {'Yes' if duplicate_frames else 'No'}")
        
        if not ask_yes_no_question("Apply Translations?", apply_msg):
            return

        # Apply translations
        try:
            logf2 = open(log_path, "a", encoding="utf-8")
        except Exception:
            logf2 = None

        write_log_header(logf2, f"{mode_label} (APPLY)", tgt, False)

        translated2, skipped2, fatal2, translated_words2, _ = process_frames(
            frames, frame_pages, tgt, False, logf2, duplicate_frames
        )

        if logf2:
            try:
                logf2.write(f"Summary (apply): translated_frames={translated2}, "
                           f"skipped_frames={skipped2}, translated_words={translated_words2}\n")
                logf2.close()
            except Exception:
                pass

        msg = (f"Translations applied successfully!\n\n"
              f"Frames translated: {translated2}\n"
              f"Words translated: {translated_words2}\n"
              f"Frames skipped: {skipped2}\n\n"
              f"Details: {log_path}")
        scribus.messageBox("Translation Complete", msg)
        return

    # Direct translate flow
    if fatal:
        msg = (f"Translation finished with error.\n\n"
              f"Frames translated: {translated}\n"
              f"Words translated: {translated_words}\n"
              f"Frames skipped: {skipped}\n\n"
              f"Details: {log_path}")
        scribus.messageBox("Translation Error", msg)
    else:
        msg = (f"Translation completed successfully!\n\n"
              f"Frames translated: {translated}\n"
              f"Words translated: {translated_words}\n"
              f"Frames skipped: {skipped}\n\n"
              f"Details: {log_path}")
        scribus.messageBox("Translation Complete", msg)


if __name__ == "__main__":
    main()
