#!/usr/bin/env python
# -*- coding: utf-8 -*-

"""
DeepL Translation Script for Scribus – Auto Source Version

What it does:
- Lets DeepL automatically detect the source language per frame
- You only choose:
    - What to translate (selected / current page / whole document)
    - Target language
- Translates all applicable text frames in scope
- Preserves:
    - Font
    - Font size
    - Text color
    - Line spacing
    - Alignment
    - Paragraph style
    - Base character style
    - Word-level character styles (e.g. bold words)
- Optional preview mode
- Logfile: scribus_translation_log.txt (next to the .sla document)
"""

import scribus
import urllib.request
import urllib.parse
import urllib.error
import json
import os
from datetime import datetime

# -----------------------------
# USER SETTINGS
# -----------------------------

DEEPL_API_KEY = "YOUR_KEY_HERE"   # <-- fill in your DeepL API key
DEEPL_URL = "https://api-free.deepl.com/v2/translate"

SUPPORTED_LANGS = {
    "EN": "EN",
    "NL": "NL",
    "FR": "FR",
    "DE": "DE",
    "ES": "ES",
    "IT": "IT",
}

# Runtime cache for translations: (tgt, text) -> translated_text
translation_cache = {}


# -----------------------------
# UTILITIES
# -----------------------------

def get_logfile_path():
    """Get a logfile path next to the current .sla document, or in CWD."""
    try:
        doc_path = scribus.getDocName()
    except Exception:
        doc_path = ""
    if not doc_path:
        return "scribus_translation_log.txt"
    base_dir = os.path.dirname(doc_path)
    if not base_dir:
        return "scribus_translation_log.txt"
    return os.path.join(base_dir, "scribus_translation_log.txt")


def write_log_header(logf, mode, tgt, preview):
    now = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
    logf.write("\n")
    logf.write("=== Scribus DeepL Translation Run (Auto Source) ===\n")
    logf.write("Time: {}\n".format(now))
    logf.write("Mode: {}\n".format(mode))
    logf.write("Target language: {}\n".format(tgt))
    logf.write("Preview only: {}\n".format("YES" if preview else "NO"))
    logf.write("-------------------------------------\n")


def log_frame(logf, status, page, frame_name, snippet_orig, snippet_new):
    """
    Log info about one frame.
    status: PREVIEW-TRANSLATE / PREVIEW-SKIP / TRANSLATED / SKIPPED / ERROR
    """
    if logf is None:
        return
    try:
        logf.write(
            "[{status}] page={page}, frame='{frame}'\n"
            "   original:  {orig}\n"
            "   translated:{new}\n".format(
                status=status,
                page=page,
                frame=frame_name,
                orig=(snippet_orig or "").replace("\n", " "),
                new=(snippet_new or "").replace("\n", " "),
            )
        )
    except Exception:
        pass


def short_snippet(text, length=50):
    if not text:
        return ""
    t = text.strip().replace("\r", " ").replace("\n", " ")
    if len(t) <= length:
        return t
    return t[:length] + "..."


# -----------------------------
# HTTP / DEEPL HELPER
# -----------------------------

def deepl_request(params):
    """
    Low-level DeepL HTTP helper.
    Returns JSON dict on success,
    returns None on fatal error (401/403/429)
    and shows a message to the user.
    """
    if not DEEPL_API_KEY or DEEPL_API_KEY == "YOUR_KEY_HERE":
        scribus.messageBox(
            "DeepL Error",
            "Please set your DeepL API key in the script.",
            scribus.ICON_WARNING
        )
        return None

    params["auth_key"] = DEEPL_API_KEY
    data = urllib.parse.urlencode(params).encode("utf-8")

    try:
        req = urllib.request.Request(DEEPL_URL, data=data)
        f = urllib.request.urlopen(req)
        js = json.loads(f.read().decode("utf-8"))
        return js

    except urllib.error.HTTPError as e:
        code = e.code
        if code == 429:
            scribus.messageBox(
                "DeepL limit reached",
                "DeepL HTTP 429: Too many requests / quota used.\n"
                "Translation will stop.",
                scribus.ICON_WARNING
            )
            return None
        if code in (401, 403):
            scribus.messageBox(
                "DeepL authentication / permission error",
                (
                    "DeepL HTTP %d.\n"
                    "This usually means:\n"
                    "- Invalid API key\n"
                    "- Wrong endpoint (free vs pro)\n"
                    "- Or your account has no access.\n"
                    "Translation will stop."
                ) % code,
                scribus.ICON_WARNING
            )
            return None
        scribus.messageBox("DeepL HTTP Error", str(e), scribus.ICON_WARNING)
        return None

    except Exception as e:
        scribus.messageBox("DeepL Error", str(e), scribus.ICON_WARNING)
        return None


# -----------------------------
# TRANSLATION ENGINE (AUTO SOURCE)
# -----------------------------

def translate_text_auto_source(text, tgt):
    """
    Call DeepL for translation of one string.
    Let DeepL auto-detect the source language.
    Uses runtime cache to avoid duplicate calls.
    Returns None on fatal DeepL error.
    """
    if not text:
        return text

    tgt_code = SUPPORTED_LANGS.get(tgt)
    if not tgt_code:
        scribus.messageBox(
            "Error",
            "Invalid target language code.",
            scribus.ICON_WARNING
        )
        return text

    cache_key = (tgt_code, text)
    if cache_key in translation_cache:
        return translation_cache[cache_key]

    params = {
        "text": text,
        "target_lang": tgt_code
        # No "source_lang" → DeepL auto-detects source per request
    }

    js = deepl_request(params)
    if js is None:
        return None

    try:
        translated = js["translations"][0]["text"]
        translation_cache[cache_key] = translated
        return translated
    except Exception as e:
        scribus.messageBox("DeepL Parse Error", str(e), scribus.ICON_WARNING)
        return text


# -----------------------------
# WORD + STYLE HELPERS
# -----------------------------

def get_words_with_positions(text):
    """
    Split text into words by whitespace.
    Return list of (start_index, length) per word.
    """
    words = []
    start = None
    for i, ch in enumerate(text):
        if ch.isspace():
            if start is not None:
                words.append((start, i - start))
                start = None
        else:
            if start is None:
                start = i
    if start is not None:
        words.append((start, len(text) - start))
    return words


def detect_word_styles(frame_name, text, base_cstyle):
    """
    For each word, detect if it has a uniform character style
    different from the base style. Returns: word_index -> style_name.
    """
    style_map = {}

    try:
        total_len = scribus.getTextLength(frame_name)
    except Exception:
        total_len = len(text)

    char_styles = []
    for i in range(total_len):
        try:
            scribus.selectText(i, 1, frame_name)
            s = scribus.getCharacterStyle(frame_name)
        except Exception:
            s = base_cstyle
        char_styles.append(s)

    # reset selection
    try:
        scribus.selectText(0, 0, frame_name)
    except Exception:
        pass

    word_positions = get_words_with_positions(text)

    for idx, (start, length) in enumerate(word_positions):
        if start + length > len(char_styles):
            continue
        styles_in_word = char_styles[start:start + length]
        if not styles_in_word:
            continue
        first_style = styles_in_word[0]
        if all(s == first_style for s in styles_in_word):
            if first_style and first_style != base_cstyle:
                style_map[idx] = first_style

    return style_map


def apply_word_styles(frame_name, text, word_style_map):
    """
    Apply character styles from word_style_map (word_index -> style_name)
    to corresponding words in the given text.
    """
    word_positions = get_words_with_positions(text)

    for idx, (start, length) in enumerate(word_positions):
        if idx not in word_style_map:
            continue
        style_name = word_style_map[idx]
        try:
            scribus.selectText(start, length, frame_name)
            scribus.setCharacterStyle(style_name, frame_name)
        except Exception:
            continue

    # clear selection
    try:
        scribus.selectText(0, 0, frame_name)
    except Exception:
        pass


# -----------------------------
# FRAME COLLECTION HELPERS
# -----------------------------

def get_frames_current_page():
    """Return (frames, frame_pages) for the current page."""
    frames = []
    frame_pages = {}
    page = scribus.currentPage()
    for item in scribus.getPageItems():
        if item[1] == 4:  # TextFrame
            name = item[0]
            frames.append(name)
            frame_pages[name] = page
    return frames, frame_pages


def get_all_text_frames_document():
    """Return (frames, frame_pages) for all pages in the document."""
    frames = []
    frame_pages = {}
    try:
        total_pages = scribus.pageCount()
    except Exception:
        return frames, frame_pages

    current = scribus.currentPage()
    for p in range(1, total_pages + 1):
        scribus.gotoPage(p)
        for item in scribus.getPageItems():
            if item[1] == 4:
                name = item[0]
                frames.append(name)
                frame_pages[name] = p
    try:
        scribus.gotoPage(current)
    except Exception:
        pass
    return frames, frame_pages


def get_selected_text_frames_with_pages():
    """
    Return (frames, frame_pages) for selected text frames.
    Uses a document-wide page map.
    """
    frames = []
    frame_pages = {}

    # Build a page map for all text frames
    all_frames, all_pages = get_all_text_frames_document()

    sel_count = scribus.selectionCount()
    if sel_count == 0:
        return frames, frame_pages

    for i in range(sel_count):
        try:
            name = scribus.getSelectedObject(i)
        except Exception:
            continue
        try:
            if scribus.getObjectType(name) == "TextFrame":
                frames.append(name)
                frame_pages[name] = all_pages.get(name, 0)
        except Exception:
            continue

    return frames, frame_pages


# -----------------------------
# FRAME PROCESSING
# -----------------------------

def process_frames(frames, frame_pages, tgt, preview, logf):
    """
    Translate or preview given frames.
    DeepL auto-detects source language.
    Returns (translated_count, skipped_count, fatal_error).
    """
    translated = 0
    skipped = 0
    fatal_error = False

    for frame_name in frames:
        page = frame_pages.get(frame_name, 0)

        text = scribus.getText(frame_name)
        if not text or not text.strip():
            skipped += 1
            status = "PREVIEW-SKIP-EMPTY" if preview else "SKIPPED-EMPTY"
            log_frame(logf, status, page, frame_name, "", "")
            continue

        snippet_orig = short_snippet(text)

        if preview:
            # Only log what WOULD be translated
            log_frame(
                logf,
                "PREVIEW-TRANSLATE",
                page,
                frame_name,
                snippet_orig,
                "<no changes in preview>"
            )
            continue

        # Remember base styles / font / color / line spacing / alignment
        try:
            base_font = scribus.getFont(frame_name)
        except Exception:
            base_font = None

        try:
            base_size = scribus.getFontSize(frame_name)
        except Exception:
            base_size = None

        try:
            base_color = scribus.getTextColor(frame_name)
        except Exception:
            base_color = None

        try:
            base_linespacing = scribus.getLineSpacing(frame_name)
        except Exception:
            base_linespacing = None

        try:
            base_align = scribus.getAlignment(frame_name)
        except Exception:
            base_align = None

        try:
            base_pstyle = scribus.getParagraphStyle(frame_name)
        except Exception:
            base_pstyle = None

        try:
            base_cstyle = scribus.getCharacterStyle(frame_name)
        except Exception:
            base_cstyle = ""
        if base_cstyle is None:
            base_cstyle = ""

        # Detect word-level styles in source text
        word_style_map = detect_word_styles(frame_name, text, base_cstyle)

        # Perform translation (DeepL auto-detects source)
        new_text = translate_text_auto_source(text, tgt)
        if new_text is None:
            # Fatal DeepL error
            fatal_error = True
            log_frame(
                logf,
                "ERROR-DEEPL",
                page,
                frame_name,
                snippet_orig,
                ""
            )
            break

        scribus.setText(new_text, frame_name)
        snippet_new = short_snippet(new_text)

        # Re-apply base font / size / color / line spacing / alignment / styles
        try:
            total_len = scribus.getTextLength(frame_name)
        except Exception:
            total_len = len(new_text)

        if total_len > 0:
            try:
                scribus.selectText(0, total_len, frame_name)
            except Exception:
                pass

            if base_font:
                try:
                    scribus.setFont(base_font, frame_name)
                except Exception:
                    pass

            if base_size is not None:
                try:
                    scribus.setFontSize(base_size, frame_name)
                except Exception:
                    pass

            if base_color:
                try:
                    scribus.setTextColor(base_color, frame_name)
                except Exception:
                    pass

            if base_linespacing is not None:
                try:
                    scribus.setLineSpacing(base_linespacing, frame_name)
                except Exception:
                    pass

            if base_align is not None:
                try:
                    scribus.setAlignment(base_align, frame_name)
                except Exception:
                    pass

            if base_pstyle:
                try:
                    scribus.setParagraphStyle(base_pstyle, frame_name)
                except Exception:
                    pass

            if base_cstyle:
                try:
                    scribus.setCharacterStyle(base_cstyle, frame_name)
                except Exception:
                    pass

            # clear selection
            try:
                scribus.selectText(0, 0, frame_name)
            except Exception:
                pass

        # Apply word-level styles on translated text
        apply_word_styles(frame_name, new_text, word_style_map)

        translated += 1
        log_frame(
            logf,
            "TRANSLATED",
            page,
            frame_name,
            snippet_orig,
            snippet_new
        )

    return translated, skipped, fatal_error


# -----------------------------
# MENUS (NUMBER-BASED)
# -----------------------------

def choose_action():
    """
    Step 0 – What do you want to do?
    [1] Test DeepL
    [2] Translate
    [3] Cancel
    """
    text = (
        "What do you want to do?\n\n"
        "[1] Test DeepL connection (small test request)\n"
        "[2] Translate text frames (auto-detect source language)\n"
        "[3] Cancel\n\n"
        "Type 1, 2 or 3:"
    )
    choice = scribus.valueDialog(
        "Action Menu",
        text,
        "2"
    )
    if not choice:
        return None
    choice = choice.strip()
    if choice == "1":
        return "test"
    if choice == "2":
        return "translate"
    return None


def choose_mode():
    """
    Step 1 – What to translate?
    [1] Current page
    [2] Selected text frames
    [3] Whole document
    [4] Cancel
    """
    text = (
        "Step 1/3 – What do you want to translate?\n\n"
        "[1] Current page (all text frames on this page)\n"
        "[2] Selected text frames only\n"
        "[3] Whole document (all pages, all text frames)\n"
        "[4] Cancel\n\n"
        "Type 1, 2, 3 or 4:"
    )
    choice = scribus.valueDialog(
        "Translation Mode",
        text,
        "2"
    )
    if not choice:
        return None
    choice = choice.strip()
    if choice == "1":
        return "page"
    if choice == "2":
        return "selected"
    if choice == "3":
        return "document"
    return None


def choose_target_language():
    text = (
        "Step 2/3 – Choose target language code:(\n\n"
        "EN = English\n"
        "NL = Dutch\n"
        "FR = French\n"
        "DE = German\n"
        "ES = Spanish\n"
        "IT = Italian\n"
    )
    lang = scribus.valueDialog(
        "Target Language",
        text,
        "FR"
    )
    if not lang:
        return None
    lang = lang.strip().upper()
    if lang in SUPPORTED_LANGS:
        return lang
    scribus.messageBox(
        "Error",
        "Invalid language code. Use EN, NL, FR, DE, ES, or IT.",
        scribus.ICON_WARNING
    )
    return None


def choose_preview():
    """
    Preview mode: show what would be translated, or translate now.
    """
    text = (
        "Step 3/3 – Translation execution\n\n"
        "[1] Preview only (no changes, just log/summary)\n"
        "[2] Translate now\n\n"
        "Type 1 or 2:"
    )
    choice = scribus.valueDialog(
        "Preview or Translate",
        text,
        "2"
    )
    if not choice:
        return False
    choice = choice.strip()
    if choice == "1":
        return True
    return False


# -----------------------------
# MAIN
# -----------------------------

def main():
    if not scribus.haveDoc():
        scribus.messageBox(
            "Error",
            "Open a document first.",
            scribus.ICON_WARNING
        )
        return

    action = choose_action()
    if action is None:
        return

    if action == "test":
        # Simple test with DeepL, also auto-detected source
        params = {
            "text": "Test",
            "target_lang": "EN"
        }
        js = deepl_request(params)
        if js is None:
            return
        try:
            translated = js["translations"][0]["text"]
        except Exception:
            translated = "<could not read DeepL response>"
        scribus.messageBox(
            "DeepL Test",
            "DeepL connection seems OK.\n"
            "Sample translation result:\n\n%s" % translated,
            scribus.ICON_INFORMATION
        )
        return

    # action == "translate"
    mode = choose_mode()
    if mode is None:
        return

    tgt = choose_target_language()
    if not tgt:
        return

    preview = choose_preview()

    # Collect frames
    if mode == "page":
        frames, frame_pages = get_frames_current_page()
        mode_label = "Current page"
    elif mode == "selected":
        frames, frame_pages = get_selected_text_frames_with_pages()
        mode_label = "Selected frames"
    elif mode == "document":
        frames, frame_pages = get_all_text_frames_document()
        mode_label = "Whole document"
    else:
        return

    if not frames:
        scribus.messageBox(
            "Info",
            "No text frames found for the selected mode.",
            scribus.ICON_INFORMATION
        )
        return

    # Open logfile
    log_path = get_logfile_path()
    try:
        logf = open(log_path, "a", encoding="utf-8")
    except Exception:
        logf = None

    write_log_header(logf, mode_label, tgt, preview)

    # Process
    translated, skipped, fatal = process_frames(
        frames,
        frame_pages,
        tgt=tgt,
        preview=preview,
        logf=logf
    )

    if logf is not None:
        try:
            logf.write(
                "Summary: translated={}, skipped={}, fatal_error={}\n".format(
                    translated, skipped, fatal
                )
            )
            logf.write("-------------------------------------\n")
            logf.close()
        except Exception:
            pass

    if preview:
        msg = (
            "Preview finished.\n\n"
            "Frames that WOULD be translated: {}\n"
            "Frames that WOULD be skipped (empty frames): {}\n\n"
            "Details were written to:\n{}"
        ).format(translated, skipped, log_path)
    else:
        if fatal:
            msg = (
                "Translation finished with a fatal DeepL error.\n\n"
                "Frames translated before error: {}\n"
                "Frames skipped (empty frames): {}\n\n"
                "See logfile for details:\n{}"
            ).format(translated, skipped, log_path)
        else:
            msg = (
                "Translation completed.\n\n"
                "Frames translated: {}\n"
                "Frames skipped (empty frames): {}\n\n"
                "Details were written to:\n{}"
            ).format(translated, skipped, log_path)

    scribus.messageBox("Done", msg, scribus.ICON_INFORMATION)


if __name__ == "__main__":
    main()

